' DataWindowPlugIn.vb
'
' This file implements an IDataWindowPlugIn-derived class.
' The plug-in implements the basic parts of a data window
' in the TLA. It has a data display window and a System
' Window Control.
' 
' The window in this sample does not show any real data.
' The purpose is to demonstrate the basics of how to 
' implement the standard parts of a data window. Any custom
' timing diagrams or data interpretations are left as
' an exercise for the plug-in developer.
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

Imports System
Imports Tektronix.LogicAnalyzer.PlugIn
Imports Tektronix.LogicAnalyzer.TpiNet
Imports Tektronix.LogicAnalyzer.Common
Imports System.Windows.Forms
Imports System.Collections

' The PlugInIdentity Attribute used below defines the text
' used in the New Data Window dialog to identify this kind
' of data window before an instance of it is created.
<PlugInIdentity("VB Sample Data Window")> _
Public Class VBDataWindowPlugIn
    Implements IDataWindowPlugIn

    ' Data member declarations.
    Dim window As DataDisplay = Nothing
    Dim instanceIsValid As Boolean = True
    Dim instanceIsGarbage As Boolean = False
    Dim theUserName As String
    Dim control As SystemWindowUserControl

    ' The default constructor required for plug-in implementations.
    Public Sub New()
    End Sub

    ' IPlugIn Members
    '----------------

    ' Initialize() is called by the TLA immediately after the 
    ' plug-in is instantiated with its constructor.
    Public Function Initialize(ByVal support As ITlaPlugInSupport, ByVal userInit As Boolean) As Boolean Implements IPlugIn.Initialize
        window = New DataDisplay()
        support.TopLevelFormToChild(window)
        control = New SystemWindowUserControl()

        AddHandler window.Closed, AddressOf OnWindowClosed
        AddHandler control.activateButton.Click, AddressOf OnActivateButton
        AddHandler control.userNameBox.KeyDown, AddressOf OnKeyDown
        AddHandler control.userNameBox.LostFocus, AddressOf OnLostFocus

        theUserName = "VBSample 1"
        UserName = theUserName
        control.userNameBox.Text = theUserName
        window.Text = theUserName

        Return True
    End Function


    ' The primary Form is the data window.
    Public ReadOnly Property PrimaryForm() As Form Implements IPlugIn.PrimaryForm
        Get
            Return window
        End Get
    End Property


    ' IValidity Members
    ' -----------------

    ' This event is raised whenever either of the two 
    ' properties below change.
    Public Event ValidityChanged As EventHandler _
        Implements IPlugIn.ValidityChanged


    ' Property that indicates whether the plug-in is 
    ' valid to be part of the system.
    Public ReadOnly Property IsValid() As Boolean Implements IPlugIn.IsValid
        Get
            Return instanceIsValid
        End Get
    End Property


    ' Property that indicates whether the plug-in is permanently
    ' invalid. This value should always be set to true when a
    ' plug-in is disposed.
    Public ReadOnly Property IsGarbage() As Boolean Implements IPlugIn.IsGarbage
        Get
            Return instanceIsGarbage
        End Get
    End Property


    ' IDisposable Member
    ' ------------------

    ' This method invalidates the plug-in and disposes 
    ' of its window and user control.
    Public Sub Dispose() Implements IPlugIn.Dispose
        If Not Nothing Is window Then
            RemoveHandler window.Closed, AddressOf Me.OnWindowClosed
            window.Dispose()
            window = Nothing
        End If

        If Not Nothing Is control Then
            RemoveHandler control.activateButton.Click, AddressOf OnActivateButton
            RemoveHandler control.userNameBox.KeyDown, AddressOf OnKeyDown
            RemoveHandler control.userNameBox.LostFocus, AddressOf OnLostFocus
            control.Dispose()
            control = Nothing
        End If

        instanceIsValid = False
        instanceIsGarbage = True

        RaiseEvent ValidityChanged(Me, EventArgs.Empty)
    End Sub


    ' IDataWindowPlugIn Members
    ' -------------------------

    ' Event that is raised when the UserName property changes.
    Public Event UserNameChanged As EventHandler _
        Implements IDataWindowPlugIn.UserNameChanged


    ' The UserName property sets or gets the name that
    ' identifies the data window to users.
    Public Property UserName() As String Implements IDataWindowPlugIn.UserName
        Get
            Return theUserName
        End Get
        Set(ByVal Value As String)
            ' First check that the name really changed
            If 0 <> String.Compare(Value, theUserName) Then
                theUserName = Value

                ' Update the control and the data window title.
                If Not window Is Nothing Then
                    window.Text = theUserName
                End If

                If Not control Is Nothing Then
                    control.userNameBox.Text = theUserName
                End If

                ' Send notification that the name changed.
                RaiseEvent UserNameChanged(Me, EventArgs.Empty)
            End If
        End Set
    End Property


    ' Returns the data windows currently used by the data window.
    ' Since this window does not actively use system data sources,
    ' it returns an empty array.
    Public Function GetDataSources() As IDataSource() Implements IDataWindowPlugIn.GetDataSources
        Dim sources As IDataSource()
        sources = New IDataSource() {}
        Return sources
    End Function


    ' The value of the SystemWindowControl property is the
    ' Control that represents the data window in the
    ' TLA System window.
    Public ReadOnly Property SystemWindowControl() As UserControl Implements IDataWindowPlugIn.SystemWindowControl
        Get
            Return control
        End Get
    End Property


    ' Event Handlers
    ' --------------

    ' The plug-in subscribes to DataDisplay.Closed. This event handler 
    ' is called when the event occurs. The plug-in responds by disposing 
    ' itself, which deletes the plug-in instance from the system.
    Private Sub OnWindowClosed(ByVal sender As System.Object, ByVal e As System.EventArgs)
        Me.Dispose()
    End Sub


    ' Changes the UserName if the Enter key is pressed in the 
    ' SystemWindowUserControl text box.
    Private Sub OnKeyDown(ByVal sender As Object, ByVal args As KeyEventArgs)
        If Keys.Enter = args.KeyCode Then
            UserName = control.userNameBox.Text
        End If
    End Sub


    ' Changes the UserName when the SystemWindowUserControl text box
    ' loses the input focus.
    Private Sub OnLostFocus(ByVal sender As Object, ByVal args As EventArgs)
        UserName = control.userNameBox.Text
    End Sub


    ' Activates the data window when the user presses the "Show Window"
    ' button on the SystemWindowUserControl.
    Private Sub OnActivateButton(ByVal sender As Object, ByVal args As EventArgs)
        If Not Nothing Is window Then
            window.Activate()
        End If
    End Sub

End Class
